unit untDemo;

// This program demonstrates some of the features and uses of FeedForDev
// FeedForDev makes it easy for a developer to use RSS feeds inside
// their applications, for continuing communication with their existing
// customers, among other things.
//
// Please let us know if any part of this demo could be made more clear
// or coded in a more understandable manner (support@feedfordev.com)

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, FeedForDev, ExtCtrls;

type
  TfrmDemo = class(TForm)
    grpFeed: TGroupBox;
    edtRssFeedUrl: TEdit;
    lblRssFeedUrl: TLabel;
    grpExamples: TGroupBox;
    btnVersion: TButton;
    btnTricks: TButton;
    btnNews: TButton;
    btnSpecials: TButton;
    grpAdvanced: TGroupBox;
    stbStatus: TStatusBar;
    btnGetFeedItems: TButton;
    btnRawFeed: TButton;
    btnShowItems: TButton;
    lblRegMessage: TLabel;
    rgpPDSortOptions: TRadioGroup;
    rgpRetrievalOptions: TRadioGroup;
    grbFiltering: TGroupBox;
    lblCategory: TLabel;
    edtCategory: TEdit;
    lblStartDate: TLabel;
    dtpStartDate: TDateTimePicker;
    lblEndDate: TLabel;
    dtpEndDate: TDateTimePicker;
    ffdRSSReader1: TffdRSSReader;
    procedure btnGetFeedItemsClick(Sender: TObject);
    procedure ffdRSSReader1Finished(Sender: TObject);
    procedure edtRssFeedUrlExit(Sender: TObject);
    procedure btnVersionClick(Sender: TObject);
    procedure btnTricksClick(Sender: TObject);
    procedure btnSpecialsClick(Sender: TObject);
    procedure btnNewsClick(Sender: TObject);
    procedure btnRawFeedClick(Sender: TObject);
    procedure btnShowItemsClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmDemo: TfrmDemo;

implementation

{$R *.dfm}

uses
  untMemo;

const
  ProgramVersion = '1.0';

var
  FeedIsLoaded: boolean;

procedure TfrmDemo.btnGetFeedItemsClick(Sender: TObject);
// Retrieves feed if necessary and loads in current items
// according to category and date settings
begin
  // load settings from screen into control
  ffdRSSReader1.Address := Trim(edtRssFeedUrl.Text);
  // set retrieval behavior
  ffdRSSReader1.Blocking := (rgpRetrievalOptions.ItemIndex = 0); // 0 - blocking, 1 - events
  // set sort order for retrieved items
  case rgpPDSortOptions.ItemIndex of
    0: ffdRSSReader1.PubDateSort := pdsNone; // no sorting
    1: ffdRSSReader1.PubDateSort := pdsAscending; // sort by pubdate, ascending
    2: ffdRSSReader1.PubDateSort := pdsDescending; // sort by pubdate, descending
  end;
  if not FeedIsLoaded then begin
    // The RSS feed entered into the address field hasn't been loaded
    ffdRSSReader1.GetFeed;
    // if the control is set to use non-blocking retrieval (event drived)
    // then we are done here, because the data will be read when the
    // OnFinished event is fired by the control.  Otherwise read the
    // resulting data now
    if ffdRSSReader1.Blocking then begin
      // blocking, so retrieval is done by now
      // first check for errors
      if ffdRSSReader1.ErrorCode <> ecNone then begin
        // got an error downloading the feed
        // update the status bar
        stbStatus.Panels.Items[0].Text := 'Status: Error';
        stbStatus.Panels.Items[1].Text := 'Current Items: 0';
        stbStatus.Panels.Items[2].Text := 'Total Items: 0';
        if ffdRSSReader1.Registered then begin
          stbStatus.Panels.Items[3].Text := 'License: Registered';
        end
        else begin
          stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
        end;
        // show message about failure
        ShowMessage('Opening the RSS feed gave the following error: ' + ffdRSSReader1.ErrorDescription);
      end
      else begin
        // RSS feed sucessfully retrieved and parsed
        // update the status bar
        stbStatus.Panels.Items[0].Text := 'Status: Loaded';
        stbStatus.Panels.Items[2].Text := 'Total Items: ' + IntToStr(ffdRSSReader1.ItemCount);
        if ffdRSSReader1.Registered then begin
          stbStatus.Panels.Items[3].Text := 'License: Registered';
        end
        else begin
          stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
        end;
        // now use the rest of the settings from the screen to filter which items
        // should be currently available
        ffdRSSReader1.CategoryFilter := Trim(edtCategory.Text);
        ffdRSSReader1.PubDateStart := dtpStartDate.Date;
        ffdRSSReader1.PubDateEnd := dtpEndDate.Date;
        // now ask for all the items in the feed that meet the criteria
        ffdRSSReader1.GetItems;
        // update the status bar
        stbStatus.Panels.Items[1].Text := 'Current Items: ' + IntToStr(ffdRSSReader1.ItemCount);
      end;
    end;
    FeedIsLoaded := True;
  end
  else begin
    // the feed has already been loaded previously, so we just need to use the settings
    // from the screen to re-filter the items
    ffdRSSReader1.CategoryFilter := Trim(edtCategory.Text);
    ffdRSSReader1.PubDateStart := dtpStartDate.Date;
    ffdRSSReader1.PubDateEnd := dtpEndDate.Date;
    // now ask for all the items in the feed that meet the criteria
    ffdRSSReader1.GetItems;
    // update the status bar
    stbStatus.Panels.Items[1].Text := 'Current Items: ' + IntToStr(ffdRSSReader1.ItemCount);
  end;
end;

procedure TfrmDemo.ffdRSSReader1Finished(Sender: TObject);
// When the control's "Blocking" property is set to false,
// it runs in an event-driven mode and returns immediately
// from the "GetFeed" call.  This event is fired when the
// retrieval of the feed is finished
begin
  // first check for errors
  if ffdRSSReader1.ErrorCode <> ecNone then begin
    // got an error downloading the feed
    // update the status bar
    stbStatus.Panels.Items[0].Text := 'Status: Error';
    stbStatus.Panels.Items[1].Text := 'Current Items: 0';
    stbStatus.Panels.Items[2].Text := 'Total Items: 0';
    if ffdRSSReader1.Registered then begin
      stbStatus.Panels.Items[3].Text := 'License: Registered';
    end
    else begin
      stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
    end;
    // show message about failure
    ShowMessage('Opening the RSS feed gave the following error: ' + ffdRSSReader1.ErrorDescription);
  end
  else begin
    // RSS feed sucessfully retrieved and parsed
    // update the status bar
    stbStatus.Panels.Items[0].Text := 'Status: Loaded';
    stbStatus.Panels.Items[2].Text := 'Total Items: ' + IntToStr(ffdRSSReader1.ItemCount);
    if ffdRSSReader1.Registered then begin
      stbStatus.Panels.Items[3].Text := 'License: Registered';
    end
    else begin
      stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
    end;
    // now use the rest of the settings from the screen to filter which items
    // should be currently available
    ffdRSSReader1.CategoryFilter := Trim(edtCategory.Text);
    ffdRSSReader1.PubDateStart := dtpStartDate.Date;
    ffdRSSReader1.PubDateEnd := dtpEndDate.Date;
    // now ask for all the items in the feed that meet the criteria
    ffdRSSReader1.GetItems;
    // update the status bar
    stbStatus.Panels.Items[1].Text := 'Current Items: ' + IntToStr(ffdRSSReader1.ItemCount);
  end;
end;

procedure TfrmDemo.edtRssFeedUrlExit(Sender: TObject);
begin
  // check to see if the url of the feed has changed
  if TEdit(Sender).Modified then begin
    // the url has been changed, so reset the status bar values
    stbStatus.Panels.Items[0].Text := 'Status: No Feed';
    stbStatus.Panels.Items[1].Text := 'Current Items: 0';
    stbStatus.Panels.Items[2].Text := 'Total Items: 0';
    if ffdRSSReader1.Registered then begin
      stbStatus.Panels.Items[3].Text := 'License: Registered';
    end
    else begin
      stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
    end;
  end;
  // set feed loaded status to false
  FeedIsLoaded := False;
  // reset the modifed flag
  TEdit(Sender).Modified := False;
end;

procedure TfrmDemo.btnVersionClick(Sender: TObject);
// sets the category to "version" to retrieve any entries
// in the feed that are version entries, and then compares
// it to the current demo program version
begin
  // set the category
  edtCategory.Text := 'version';
  // reset date fields in case they were changed by the user
  dtpStartDate.Date := StrToDate('01/01/2001');
  dtpEndDate.Date := StrToDate('01/01/2099');
  // get matching items
  btnGetFeedItems.Click;
  // check to see if a version item was found in the feed
  if ffdRSSReader1.ItemCount > 0 then begin
    // found version entry in feed.  Compare it to the version of the
    // current program.  I am just using a constant set to 1.0 here,
    // in a real program you would do something better
    if ProgramVersion < Trim(ffdRSSReader1.ItemTitle) then begin
      // a version item was found with a newer version
      // display the text from the feed's description field
      // wich could contain info about the new version
      ShowMessage(ffdRSSReader1.ItemDescription);
    end
    else begin
      ShowMessage('Your program is up to date');
    end;
  end
  else begin
    ShowMessage('There are no version items in that feed');
  end;
end;

procedure TfrmDemo.btnTricksClick(Sender: TObject);
// sets the category to "tip" to retrieve any entries
// in the feed that are tips & tricks entries
var
  Counter: integer;
begin
  // set the category
  edtCategory.Text := 'tip';
  // reset date fields in case they were changed by the user
  dtpStartDate.Date := StrToDate('01/01/2001');
  dtpEndDate.Date := StrToDate('01/01/2099');
  // get matching items
  btnGetFeedItems.Click;
  // check to see if any items were found in the feed
  // if so, display them
  Counter := 1;
  while Counter <= ffdRSSReader1.ItemCount do begin
    // set which item to display info from
    ffdRSSReader1.ItemNumber := Counter;
    // display tip in a message box
    ShowMessage(ffdRSSReader1.ItemTitle + #13#10 + ffdRSSReader1.ItemDescription);
    // move on to the next item
    Inc(Counter);
  end;
  // if there weren't any items, then show message about that
  if Counter = 1 then begin
    ShowMessage('There weren''t any tips in the feed');
  end;
end;

procedure TfrmDemo.btnSpecialsClick(Sender: TObject);
// sets the category to "special" to retrieve any entries
// in the feed that are specials
var
  Counter: integer;
begin
  // set the category
  edtCategory.Text := 'special';
  // reset date fields in case they were changed by the user
  dtpStartDate.Date := StrToDate('01/01/2001');
  dtpEndDate.Date := StrToDate('01/01/2099');
  // get matching items
  btnGetFeedItems.Click;
  // check to see if any items were found in the feed
  // if so, display them
  Counter := 1;
  while Counter <= ffdRSSReader1.ItemCount do begin
    // set which item to display info from
    ffdRSSReader1.ItemNumber := Counter;
    // display tip in a message box
    ShowMessage(ffdRSSReader1.ItemTitle + #13#10 + ffdRSSReader1.ItemDescription + #13#10 + 'For more info go here: ' + ffdRSSReader1.ItemLink);
    // move on to the next item
    Inc(Counter);
  end;
  // if there weren't any items, then show message about that
  if Counter = 1 then begin
    ShowMessage('There weren''t any specials in the feed');
  end;
end;

procedure TfrmDemo.btnNewsClick(Sender: TObject);
// displays any news items in another form with
// a memo field
var
  Counter: integer;
begin
  // set the category
  edtCategory.Text := 'news';
  // reset date fields in case they were changed by the user
  dtpStartDate.Date := StrToDate('01/01/2001');
  dtpEndDate.Date := StrToDate('01/01/2099');
  // get matching items
  btnGetFeedItems.Click;
  // clear the memo box
  frmMemo.mmoFeedData.Clear;
  // check to see if any items were found in the feed
  // if so, display them
  Counter := 1;
  while Counter <= ffdRSSReader1.ItemCount do begin
    // set which item to display info from
    ffdRSSReader1.ItemNumber := Counter;
    // add news item to memo box
    frmMemo.mmoFeedData.Lines.Add(ffdRSSReader1.ItemTitle);
    frmMemo.mmoFeedData.Lines.Add(ffdRSSReader1.ItemDescription);
    // add blank line between items
    frmMemo.mmoFeedData.Lines.Add('');
    // move on to the next item
    Inc(Counter);
  end;
  // if there weren't any items, then show message about that
  if Counter > 1 then begin
    // show the memo form
    frmMemo.ShowModal;
  end
  else begin
    // otherwise say there were not news items
    ShowMessage('There weren''t any news items in the feed');
  end;
end;

procedure TfrmDemo.btnRawFeedClick(Sender: TObject);
// this will show the feed's raw source data
begin
  // clear the memo field
  frmMemo.mmoFeedData.Clear;
  // add the raw source to the memo field
  frmMemo.mmoFeedData.Text := ffdRSSReader1.RawFeed;
  frmMemo.ShowModal;
end;

procedure TfrmDemo.btnShowItemsClick(Sender: TObject);
// displays the title, description, and link for each item in the feed
// that matches the filter criteria (i.e. category, start and end dates)
var
  Counter: integer;
begin
  // clear memo
  frmMemo.mmoFeedData.Clear;
  // add feed's title and pubdate
  frmMemo.mmoFeedData.Lines.Add(ffdRSSReader1.FeedTitle + ' - ' +
      DateTimeToStr(ffdRSSReader1.FeedPubDate));
  frmMemo.mmoFeedData.Lines.Add('');
  Counter := 1;
  while Counter <= ffdRSSReader1.ItemCount do begin
    // set the current item
    ffdRSSReader1.ItemNumber := Counter;
    // add the item's fields to the memo box
    frmMemo.mmoFeedData.Lines.Add('Item # ' + IntToStr(Counter));
    frmMemo.mmoFeedData.Lines.Add('Title: ' + ffdRSSReader1.ItemTitle);
    frmMemo.mmoFeedData.Lines.Add('Description: ' + ffdRSSReader1.ItemDescription);
    frmMemo.mmoFeedData.Lines.Add('Link: ' + ffdRSSReader1.ItemLink);
    frmMemo.mmoFeedData.Lines.Add('PubDate: ' + DateTimeToStr(ffdRSSReader1.ItemPubDate));
    // add a blank line between items
    frmMemo.mmoFeedData.Lines.Add('');
    // move to next item
    Inc(Counter);
  end;
  if Counter > 1 then begin
    // at least one item was found, so show memo form
    frmMemo.ShowModal;
  end
  else begin
    // there aren't any current items, so say so
    ShowMessage('There aren''t any current items to display');
  end;
end;

procedure TfrmDemo.FormShow(Sender: TObject);
begin
  if ffdRSSReader1.Registered then begin
    stbStatus.Panels.Items[3].Text := 'License: Registered';
  end
  else begin
    stbStatus.Panels.Items[3].Text := 'License: Un-Registered';
  end;
end;

end.
